'use client';

import type { CartItem, Product, Addon } from '@/lib/types';
import type { ReactNode } from 'react';
import { createContext, useState, useMemo } from 'react';
import {v4 as uuidv4} from 'uuid';

export interface CartContextType {
  cartItems: CartItem[];
  addItem: (product: Product, selectedAddons: Addon[], quantity: number) => void;
  updateItemQuantity: (cartItemId: string, quantity: number) => void;
  removeItem: (cartItemId: string) => void;
  clearCart: () => void;
  totalItems: number;
  totalPrice: number;
}

export const CartContext = createContext<CartContextType | undefined>(
  undefined
);

export function CartProvider({ children }: { children: ReactNode }) {
  const [cartItems, setCartItems] = useState<CartItem[]>([]);

  const addItem = (product: Product, selectedAddons: Addon[], quantity: number) => {
    setCartItems((prevItems) => {
        const newItem: CartItem = {
            id: uuidv4(),
            product,
            quantity,
            selectedAddons
        };
      return [...prevItems, newItem];
    });
  };

  const updateItemQuantity = (cartItemId: string, quantity: number) => {
    setCartItems((prevItems) => {
      if (quantity <= 0) {
        return prevItems.filter((item) => item.id !== cartItemId);
      }
      return prevItems.map((item) =>
        item.id === cartItemId ? { ...item, quantity } : item
      );
    });
  };

  const removeItem = (cartItemId: string) => {
    setCartItems((prevItems) =>
      prevItems.filter((item) => item.id !== cartItemId)
    );
  };

  const clearCart = () => {
    setCartItems([]);
  };

  const totalItems = useMemo(() => {
    return cartItems.reduce((sum, item) => sum + item.quantity, 0);
  }, [cartItems]);

  const totalPrice = useMemo(() => {
    return cartItems.reduce(
      (sum, item) => {
        const addonsPrice = item.selectedAddons.reduce((addonSum, addon) => addonSum + addon.price, 0);
        const itemTotal = (item.product.price + addonsPrice) * item.quantity;
        return sum + itemTotal;
      },
      0
    );
  }, [cartItems]);

  const value = {
    cartItems,
    addItem,
    updateItemQuantity,
    removeItem,
    clearCart,
    totalItems,
    totalPrice,
  };

  return <CartContext.Provider value={value}>{children}</CartContext.Provider>;
}
